
//#############################################################################
//
// FILE:    CRD25DA12N-FMC_main.c
//
// TITLE:   CRD25DA12N-FMC 25 kW Controller Code
//
// AUTHOR:  WOLFSPEED
//
// PURPOSE:
//  This software is designed for the evaluation of the Wolfspeed
//  CRD25DA12N-FMC Three-Phase Inverter Reference Design using the Wolfspeed
//  CCB021M12FM3T six-pack SiC power module and provides only the basic
//  code required to create an open-loop inverter. The basic 3-phase PWM
//  parameters can be controlled over a CAN interface along with rudimentary
//  feedback from the on-board sensors. It is designed as a starting point
//  only and it is left to the user any customization for a particular
//  application.
//
// REVISION HISTORY:
// V1.0.0   Initial Release
// V2.0.0   Updated to support V2.0 hardware. Added CLA & more CAN feedback/
//          control.
//
//#############################################################################

//
// Included Files
//
#include <CRD25DA12N-FMC.h>

//
// Main
//
void main(void) {
    //
    // Initializes system control, device clock, and peripherals
    //
    Device_init();
    Interrupt_initModule();
    Interrupt_initVectorTable();
    Board_init();

    //
    //  Initialize parameters, Disable Sync, Configure EPWMs
    //
    initOperatingParameters();
    SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
    initEpwm();

    //
    // Enable sync and clock to PWM
    //
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    //
    // Assign the interrupt service routines to ePWM interrupts and enable
    //
    Interrupt_register(INT_EPWM1, &epwm1ISR);
    Interrupt_enable(INT_EPWM1);

    //
    // Enable Global Interrupt (INTM) and realtime interrupt (DBGM)
    //
    EINT;
    ERTM;

    //
    // Apply offset correction to sensors and set overcurrent protection
    //
    setSensorOffsetCorrection();
    setOvercurrentProtection(DEFAULTOVERCURRENT);

    //
    // Initialize Variables
    //
    uint16_t rxMsgData[8];

    //
    // Green LED Always On
    //
    enableOnBoardGreenLed();

    //
    // Loop Indefinitely
    //
    while(1) {
        //
        // Create 1 Second Delay
        //
        DEVICE_DELAY_US(1000000);

        //
        // Check for Controller CAN Message. If Present, Decode and Update Controller
        //
        if (CAN_readMessage(myCAN0_BASE, CANOBJECTRXCONTROL, rxMsgData)) {
            receiveCanControls(rxMsgData);
            updateClockCycleParameters();
            setAllEpwm();
        }

        //
        // Transmit/Receive Control/Feedback Parameters Via CAN
        //
        sendCanControls();
        sendCanSensors();

        //
        // Provide Visual Feedback By Toggling On-Board Yellow LED
        //
        toggleOnBoardYellowLed();
    }
}

//*****************************************************************************
//
//! EPWM1 interrupt service routine (ISR) runs once per switching period.
//! Starts ADC conversion and stores ADC measurement of one phase. Alternate
//! which phase is measured to reduce compute resources (rather than measuring
//! all on every iteration).
//!
//! \return None.
//
//*****************************************************************************
__interrupt void epwm1ISR(void)
{
    //
    // ADCs
    //
    forceCurrentSocByNumber(currentSensorNum);
    storeCurrentSensorsAlternate(currentSensorNum);

    //
    // Cycle Which Current Sensor is Measured
    //
    currentSensorNum++;
    if (currentSensorNum >= NUMPHASES)
        currentSensorNum = 0;

    //
    // Clear INT flag for this timer and Acknowledge
    //
    EPWM_clearEventTriggerInterruptFlag(EPWM1_BASE);
    Interrupt_clearACKGroup(INTERRUPT_ACK_GROUP3);
}

//*****************************************************************************
//
//! After Control Law Accelerator (CLA) Task 1 runs, increment the phase angle
//! for the sine PWM (SPWM) reference.
//!
//! \return None.
//
//*****************************************************************************
__interrupt void cla1Isr1 () {
    //
    // Increment Phase Angle
    //
    uint16_t i;
    for (i = 0; i < NUMPHASES; i++) {
        epwmAngles[i] += angleStep;
        if (epwmAngles[i] > PI2)
            epwmAngles[i] -= PI2;
    }

    //
    // Acknowledge the end-of-task interrupt for task 1
    //
    Interrupt_clearACKGroup(INTERRUPT_ACK_GROUP11);
}

//
// End of file
//
